/*
 * Fonteditfs - A full-screen console font editor.
 * Copyright (C) 2002, 2003 Uri Shaked <uri@keves.org>.
 * Portions of the code were donated by amir shalem <amir@boom.org.il>.
 * Homepage: http://fonteditfs.sourceforge.net/
 *
 * Licensed under the BSD license:
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY ITS AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * $Id: fnteditfs.c,v 1.4 2003/09/20 15:17:34 uri Exp $
 * Main source file.
 */

#include <string.h>
#include <signal.h>
#include <curses.h>
#include <fcntl.h>
#include <limits.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include "font.h"
#include "stack.h"

#define VERSION "1.2"

char *helptext[] = {
    "Left/Up/Right/Down - move",
    "Space - Toggle",
    "W - Wide editing mode",
    "L - Load file",
    "S - Save file",
    "C - Save font to console",
    "D - Load font from console",
    "Q - Quit",
    "R - redraw screen",
    "P - Push current -> stack",
    "O - pOp  stack -> current",
    "PgUp - previous char",
    "PgDown - next char",
    "# - jump to specific char"
};
const int helplines = 14;

char *title = "fnteditfs "VERSION" - A Full-Screen Console Font Editor";

int charnum = 65;
int editx = 0, edity = 0;
int wide_editing = 0;

WINDOW *helpwin, *editwin, *statusln;
char fontbuf[4096];
char fnamebuf[_POSIX_PATH_MAX+1];
Stack *char_stack;

#define edit_line_refresh \
    prefresh(pad, 0, x / (width - 10) * (width - 10) - 10, top, left, top, left + width)

char *edit_line (char *buf, int bufsize, int top, int left, int width, int numeric) {
    WINDOW *pad = newpad(1, bufsize);
    int ch, buflen = strlen(buf);
    int insertmode = 1;
    int y = 0, x = buflen;
    if (!pad)
	return (char*)0;
    keypad(pad, 1);
    wattrset(pad, A_BOLD);
    wclear(pad);
    waddstr(pad, buf);
    edit_line_refresh;
    while (1) {
	getyx(pad, y, x);
	ch = wgetch(pad);
	switch (ch) {
        case -1:
	    return NULL;
	case KEY_IC: // insert
	    insertmode = !insertmode;
	    break;
	case KEY_DC: // delete
	    if (buflen == 0) {
		beep();
		edit_line_refresh;
		break;
	    }
	    if (x == buflen)
		wmove(pad, y, --x);
	    memcpy(&buf[x], &buf[x+1], buflen - x + 1);
	    buf[--buflen] = (char)0;
	    waddstr(pad, &buf[x]);
	    wclrtoeol(pad);
	    wmove(pad, y, x);
	    edit_line_refresh;
	    break;
	case KEY_BACKSPACE: // backspace
	    if (buflen == 0 || x == 0) {
		beep();
		edit_line_refresh;
		break;
	    }
	    if (x == buflen) {
		wmove(pad, y, --x);
        	waddch(pad, ' ');
		wmove(pad, y, x);
        	buflen--;
	    } else {
		wmove(pad, y, --x);
		memcpy(&buf[x], &buf[x+1], buflen - x + 1);
		buf[--buflen] = (char)0;
		waddstr(pad, &buf[x]);
		wclrtoeol(pad);
		wmove(pad, y, x);
	    }
            edit_line_refresh;
	    break;
	case KEY_HOME: // home
	    x = 0;
	    wmove(pad, y, x);
            edit_line_refresh;
	    break;
	case KEY_END: // end
	    x = buflen;
	    wmove(pad, y, x);
            edit_line_refresh;
	    break;
	case KEY_LEFT: // left arrow
	    if (!x) { 
		beep();
                edit_line_refresh;
		break;
	    }
	    x--;
	    wmove(pad, y, x);
            edit_line_refresh;
	    break;
	case KEY_RIGHT: // right arrow
	    if (x == buflen) { 
		beep();
                edit_line_refresh;
		break;
	    }
	    x++;
	    wmove(pad, y, x);
            edit_line_refresh;
	    break;
	case 12: // ^L char
	    redrawwin(pad);
            edit_line_refresh;
	    break;
	case '\r': // CR
	case '\n': // LF (line feed)
	    wattrset(pad, 0);
	    wmove(pad, 0, 0);
	    wclear(pad);
            edit_line_refresh;
	    delwin(pad);
	    buf[buflen] = (char)0;
	    return (char*)buf;
	case 21: // ^U char
	    x = 0;
	    buflen = 0;
	    wclear(pad);
            edit_line_refresh;
	    break;
	case 27: // ESC
	    wattrset(pad, 0);
	    wmove(pad, 0, 0);
	    wclear(pad);
            edit_line_refresh;
	    delwin(pad);
	    return (char*) 0;
	default: // any other key
	    if ((ch > 255 || ch < 32) || (numeric && (ch < '0' || ch > '9'))) {
		beep();
                edit_line_refresh;
		break;
	    }
	    if (buflen >= bufsize - 1) { // end of buffer
		beep();
                edit_line_refresh;
		break;
	    }
	    if (x == buflen) {
	        buf[x] = ch;
    		waddch(pad, ch);
        	buflen++;
	    } else {
	        if (insertmode) {
		    memcpy(&buf[x+1], &buf[x], buflen - x);
	    	    buf[++buflen] = (char)0;
		    buf[x] = ch;
	    	    waddstr(pad, &buf[x]);
	    	    wmove(pad, 0, ++x);
		} else {
	    	    buf[x] = ch;
            	    waddch(pad, ch);
		}
	    }
            edit_line_refresh;
    	}
    }
}

void draw_char (WINDOW *win, int num, char buf[]) {
    int i, ch;
    wmove(win, 0, 12 / 2 - 5);
    wprintw(win, wide_editing ? " Char %03d [WIDE] " : " Char %03d ", charnum);
    for (i = 0; i < 128; i++) {
	if (i % 8 == 0)
	    wmove(win, i / 8 + 1, 2);
	ch = buf[i / 8] & (1 << (7 - i % 8)) ? ACS_BLOCK : ' ';
	waddch(win, ch);
	if (wide_editing)
	    waddch(win, ch);
    }
    wmove(win, edity + 1, editx * (wide_editing + 1) + 2);
}

void toggle_char_bit(WINDOW *win, int num, char *buf, int y, int x, int chwidth) {
    int i;
    buf[num * 16 + y] ^= (1 << (7 - x));
    wmove(win, y + 1, x * chwidth + 2);
    for (i = 0; i < chwidth; i++)
	waddch(win, buf[num * 16 + y] & (1 << (7 - x)) ? ACS_BLOCK : ' ');
    wmove(win, y + 1, x * chwidth + 2);
}

void draw_screen () {
    int i, len = strlen(title);
    attrset(A_STANDOUT);
    move(0, 0);
    for (i = 0; i < (80 - len) / 2; i++)
	addch(' ');
    addstr(title);
    for (i = (80 - len) / 2 + len; i < COLS; i++)
	addch(' ');
    helpwin = newwin(helplines + 2, 30, 5, 5);
    box(helpwin, ACS_VLINE, ACS_HLINE);
    wmove(helpwin, 0, 30 / 2 - 3);
    waddstr(helpwin, " Keys ");
    for (i = 0; i < helplines; i++) {
	wmove(helpwin, 1 + i, 2);
	waddstr(helpwin, helptext[i]);
    }
    editwin = newwin(18, 12, 5, 45);
    box(editwin, ACS_VLINE, ACS_HLINE);
    draw_char(editwin, charnum, &fontbuf[charnum * 16]);
    refresh();
    statusln = newwin(1, COLS, LINES - 1, 0);
    wrefresh(helpwin);
    wrefresh(editwin);
}

void ask_move_char () {
    char buf[4];
    buf[0] = 0;
    wclear(statusln);
    mvwaddstr(statusln, 0, 0, "New character #:");
    wrefresh(statusln);
    if (edit_line(buf, sizeof(buf), LINES - 1, 17, 4, 1)) {
	charnum = atol(buf) % 256;
        draw_char(editwin, charnum, &fontbuf[charnum * 16]);
	wrefresh(editwin);
    }
    wclear(statusln);
    wrefresh(statusln);
    wrefresh(editwin);
}

void interactive_load_file () {
    char buf[sizeof(fnamebuf)];
    int rc;
    strcpy(buf, fnamebuf);
    wclear(statusln);
    mvwaddstr(statusln, 0, 0, "Load from file:");
    wrefresh(statusln);
    if (!edit_line(buf, sizeof(buf), LINES - 1, 16, COLS - 16, 0)) {
	wclear(statusln);
        mvwaddstr(statusln, 0, 0, "Load canceled.");
        wrefresh(statusln);
	wrefresh(editwin);
	return;
    }
    strcpy(fnamebuf, buf);
    // first try to load as uuencoded font
    rc = font_uuload(buf, fontbuf, sizeof(fontbuf));
    if (rc == -1) // otherwise, try normal load
	rc = font_load(buf, fontbuf, sizeof(fontbuf));
    if (rc) {
	wclear(statusln);
        mvwprintw(statusln, 0, 0, "Load error: %s", font_error);
        wrefresh(statusln);
	wrefresh(editwin);
	return;
    }
    wclear(statusln);
    mvwaddstr(statusln, 0, 0, "Font loaded.");
    wrefresh(statusln);
    draw_char(editwin, charnum, &fontbuf[charnum * 16]);
    wrefresh(editwin);
}

void interactive_save_file () {
    char buf[sizeof(fnamebuf)];
    int rc;
    strcpy(buf, fnamebuf);
    wclear(statusln);
    mvwaddstr(statusln, 0, 0, "Save to file:");
    wrefresh(statusln);
    if (!edit_line(buf, sizeof(buf), LINES - 1, 14, COLS - 14, 0)) {
	wclear(statusln);
        mvwaddstr(statusln, 0, 0, "Save canceled.");
        wrefresh(statusln);
	wrefresh(editwin);
	return;
    }
    strcpy(fnamebuf, buf);
    rc = font_save(buf, fontbuf, sizeof(fontbuf));
    if (rc) {
	wclear(statusln);
        mvwprintw(statusln, 0, 0, "Save error: %s", font_error);
        wrefresh(statusln);
	wrefresh(editwin);
	return;
    }
    wclear(statusln);
    mvwaddstr(statusln, 0, 0, "Font saved.");
    wrefresh(statusln);
    wrefresh(editwin);
}

/**********************************/
/* operation system depended code */
/**********************************/

#if defined(__FreeBSD__)

#include <sys/ioctl.h>
#include <sys/consio.h>

int do_ioctl(const char *header, unsigned long request, char *buf) {
    int fd;
    
    if (ioctl(0, request, buf) < 0) {
	fd = open("/dev/console", O_RDONLY, 0);
	if (fd < 0) {
	    wclear(statusln);
    	    mvwprintw(statusln, 0, 0, "%s failed to open /dev/console: %s", header, strerror(errno));
	    wrefresh(statusln);
	    wrefresh(editwin);
	    return -1;
	}
	if (ioctl(fd, request, buf) < 0) {
	    wclear(statusln);
    	    mvwprintw(statusln, 0, 0, "%s failed to ioctl /dev/console: %s", header, strerror(errno));
	    wrefresh(statusln);
	    wrefresh(editwin);
	    close(fd);
	    return -1;
	}
    }
    return 0;
}

void save_font_to_console () {
    if (do_ioctl("Save error:", PIO_FONT8x16, fontbuf) < 0)
	return;

    wclear(statusln);
    mvwaddstr(statusln, 0, 0, "Font saved to console.");
    wrefresh(statusln);
    wrefresh(editwin);
}

void load_font_from_console () {
    if (do_ioctl("Load error:", GIO_FONT8x16, fontbuf) < 0)
	return;

    wclear(statusln);
    mvwaddstr(statusln, 0, 0, "Font loaded from console.");
    wrefresh(statusln);
    draw_char(editwin, charnum, &fontbuf[charnum * 16]);
    wrefresh(editwin);
}

#else

void save_font_to_console () {
    wclear(statusln);
    mvwaddstr(statusln, 0, 0, "No console interaction is supported under this OS");
    wrefresh(statusln);
    wrefresh(editwin);
}

void load_font_from_console () {
    save_font_to_console();
}

#endif /* OS is not supported */

/*****************************************/
/* end of operation system depended code */
/*****************************************/

void input_loop () {
    char *tmp;
    int c;
    while (1) {
	c = getch();
	switch (c) {
	    case KEY_UP: 
		edity ? edity -- : (edity = 15);
		wmove(editwin, edity+1, editx*(wide_editing + 1)+2);
		wrefresh(editwin);
		break;
	    case KEY_LEFT:
		editx ? editx -- : (editx = 7);
		wmove(editwin, edity+1, editx*(wide_editing + 1)+2);
		wrefresh(editwin);
		break;
	    case KEY_DOWN:
		edity ++; edity %= 16;
		wmove(editwin, edity+1, editx*(wide_editing + 1)+2);
		wrefresh(editwin);
		break;
	    case KEY_RIGHT:
		editx ++; editx %= 8;
		wmove(editwin, edity+1, editx*(wide_editing + 1)+2);
		wrefresh(editwin);
		break;
	    case KEY_PPAGE:
		charnum ? charnum-- : (charnum = 255);
	        draw_char(editwin, charnum, &fontbuf[charnum * 16]);
		wrefresh(editwin);
		break;
	    case KEY_NPAGE:
		charnum++;
		charnum %= 256;
	        draw_char(editwin, charnum, &fontbuf[charnum * 16]);
		wrefresh(editwin);
		break;
	    case '#':
		ask_move_char();
		break;
	    case ' ':
		toggle_char_bit(editwin, charnum, fontbuf, edity, editx, wide_editing + 1);
		wrefresh(editwin);
		break;
	    case 'l': case 'L':
		interactive_load_file();
		break;
	    case 's': case 'S':
		interactive_save_file();
		break;
	    case 'c': case 'C':
		save_font_to_console();
		break;
	    case 'd': case 'D':
		load_font_from_console();
		break;
	    case 'p': case 'P':
		tmp = (char*)malloc(16);
		memcpy(tmp, &fontbuf[charnum * 16], 16);
		stack_push(char_stack, tmp);
	        wclear(statusln);
	        mvwprintw(statusln, 0, 0, "Character %d pushed to stack (new stack size %d).",
			charnum, stack_size(char_stack));
	        wrefresh(statusln);
		break;
	    case 'o': case 'O':
		tmp = (char*)stack_pop(char_stack);
	        wclear(statusln);
		if (!tmp) {
		    mvwprintw(statusln, 0, 0, "Stack is already empty, can't pop.",
			charnum, stack_size(char_stack));
		    wrefresh(statusln);
		    break;
		}
		memcpy(&fontbuf[charnum * 16], tmp, 16);
	        draw_char(editwin, charnum, &fontbuf[charnum * 16]);
	        wrefresh(editwin);
	        mvwprintw(statusln, 0, 0, "Character %d poped from stack (new stack size %d).",
			charnum, stack_size(char_stack));
	        wrefresh(statusln);
		break;
	    case 'w': case 'W':
		wide_editing = !wide_editing;
		delwin(editwin);
		editwin = newwin(18, wide_editing ? 20 : 12, 5, 45);
		box(editwin, ACS_VLINE, ACS_HLINE);
	        draw_char(editwin, charnum, &fontbuf[charnum * 16]);
		if (wide_editing) {
		    wrefresh(editwin);
		    break;
		}
	    case 'r': case 'R':
		redrawwin(stdscr);
		redrawwin(helpwin);
		redrawwin(editwin);
		refresh();
		wrefresh(helpwin);
	        wrefresh(editwin);
		break;
	    case 'q': case 'Q':
		return;
	}
    }
}

void print_usage (char *argv[]) {
    printf("Full screen font editor v" VERSION " by Uri Shaked <uri@keves.org>\n");
    printf("Usage: %s fontfile\n", argv[0]);
    exit(1);
}

int main (int argc, char *argv[]) {
    int rc;
    if (argc > 2)
	print_usage(argv);
    memset(fontbuf, 0, sizeof(fontbuf));
    if (argc == 2) {
        strcpy(fnamebuf, argv[1]);
	rc = font_uuload(argv[1], fontbuf, sizeof(fontbuf));
	if (rc == -1)
	    rc = font_load(argv[1], fontbuf, sizeof(fontbuf));
	if (rc) {
	    printf("Load font: %s: %s", argv[1], font_error);
	    return 1;
	}
    } else
	fnamebuf[0] = 0;
    initscr();
    cbreak();
    noecho();
    intrflush(stdscr,FALSE);
    keypad(stdscr, TRUE);
    nonl();
    char_stack = stack_init();
    draw_screen();
    // if no font file loaded, try loading console font.
    if (fnamebuf[0] == 0)
        load_font_from_console();
    input_loop();
    endwin();
    return 0;
}
