﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Array.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/stream/ResponseStream.h>
#include <aws/kinesis-video-archived-media/KinesisVideoArchivedMedia_EXPORTS.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace KinesisVideoArchivedMedia {
namespace Model {
class GetClipResult {
 public:
  AWS_KINESISVIDEOARCHIVEDMEDIA_API GetClipResult() = default;
  AWS_KINESISVIDEOARCHIVEDMEDIA_API GetClipResult(GetClipResult&&) = default;
  AWS_KINESISVIDEOARCHIVEDMEDIA_API GetClipResult& operator=(GetClipResult&&) = default;
  // we delete these because Microsoft doesn't handle move generation correctly
  // and we therefore don't trust them to get it right here either.
  GetClipResult(const GetClipResult&) = delete;
  GetClipResult& operator=(const GetClipResult&) = delete;

  AWS_KINESISVIDEOARCHIVEDMEDIA_API GetClipResult(Aws::AmazonWebServiceResult<Aws::Utils::Stream::ResponseStream>&& result);
  AWS_KINESISVIDEOARCHIVEDMEDIA_API GetClipResult& operator=(Aws::AmazonWebServiceResult<Aws::Utils::Stream::ResponseStream>&& result);

  ///@{
  /**
   * <p>The content type of the media in the requested clip.</p>
   */
  inline const Aws::String& GetContentType() const { return m_contentType; }
  template <typename ContentTypeT = Aws::String>
  void SetContentType(ContentTypeT&& value) {
    m_contentTypeHasBeenSet = true;
    m_contentType = std::forward<ContentTypeT>(value);
  }
  template <typename ContentTypeT = Aws::String>
  GetClipResult& WithContentType(ContentTypeT&& value) {
    SetContentType(std::forward<ContentTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Traditional MP4 file that contains the media clip from the specified video
   * stream. The output will contain the first 100 MB or the first 200 fragments from
   * the specified start timestamp. For more information, see <a
   * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis
   * Video Streams Limits</a>. </p>
   */
  inline Aws::IOStream& GetPayload() const { return m_payload.GetUnderlyingStream(); }
  inline void ReplaceBody(Aws::IOStream* body) { m_payload = Aws::Utils::Stream::ResponseStream(body); }

  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  GetClipResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_contentType;

  Aws::Utils::Stream::ResponseStream m_payload{};

  Aws::String m_requestId;
  bool m_contentTypeHasBeenSet = false;
  bool m_payloadHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisVideoArchivedMedia
}  // namespace Aws
