﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/AttachmentsSource.h>
#include <aws/ssm/model/DocumentFormat.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 */
class UpdateDocumentRequest : public SSMRequest {
 public:
  AWS_SSM_API UpdateDocumentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateDocument"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A valid JSON or YAML string.</p>
   */
  inline const Aws::String& GetContent() const { return m_content; }
  inline bool ContentHasBeenSet() const { return m_contentHasBeenSet; }
  template <typename ContentT = Aws::String>
  void SetContent(ContentT&& value) {
    m_contentHasBeenSet = true;
    m_content = std::forward<ContentT>(value);
  }
  template <typename ContentT = Aws::String>
  UpdateDocumentRequest& WithContent(ContentT&& value) {
    SetContent(std::forward<ContentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of key-value pairs that describe attachments to a version of a
   * document.</p>
   */
  inline const Aws::Vector<AttachmentsSource>& GetAttachments() const { return m_attachments; }
  inline bool AttachmentsHasBeenSet() const { return m_attachmentsHasBeenSet; }
  template <typename AttachmentsT = Aws::Vector<AttachmentsSource>>
  void SetAttachments(AttachmentsT&& value) {
    m_attachmentsHasBeenSet = true;
    m_attachments = std::forward<AttachmentsT>(value);
  }
  template <typename AttachmentsT = Aws::Vector<AttachmentsSource>>
  UpdateDocumentRequest& WithAttachments(AttachmentsT&& value) {
    SetAttachments(std::forward<AttachmentsT>(value));
    return *this;
  }
  template <typename AttachmentsT = AttachmentsSource>
  UpdateDocumentRequest& AddAttachments(AttachmentsT&& value) {
    m_attachmentsHasBeenSet = true;
    m_attachments.emplace_back(std::forward<AttachmentsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the SSM document that you want to update.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateDocumentRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The friendly name of the SSM document that you want to update. This value can
   * differ for each version of the document. If you don't specify a value for this
   * parameter in your request, the existing value is applied to the new document
   * version.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  UpdateDocumentRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional field specifying the version of the artifact you are updating
   * with the document. For example, 12.6. This value is unique across all versions
   * of a document, and can't be changed.</p>
   */
  inline const Aws::String& GetVersionName() const { return m_versionName; }
  inline bool VersionNameHasBeenSet() const { return m_versionNameHasBeenSet; }
  template <typename VersionNameT = Aws::String>
  void SetVersionName(VersionNameT&& value) {
    m_versionNameHasBeenSet = true;
    m_versionName = std::forward<VersionNameT>(value);
  }
  template <typename VersionNameT = Aws::String>
  UpdateDocumentRequest& WithVersionName(VersionNameT&& value) {
    SetVersionName(std::forward<VersionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the document that you want to update. Currently, Systems
   * Manager supports updating only the latest version of the document. You can
   * specify the version number of the latest version or use the <code>$LATEST</code>
   * variable.</p>  <p>If you change a document version for a State Manager
   * association, Systems Manager immediately runs the association unless you
   * previously specifed the <code>apply-only-at-cron-interval</code> parameter.</p>
   *
   */
  inline const Aws::String& GetDocumentVersion() const { return m_documentVersion; }
  inline bool DocumentVersionHasBeenSet() const { return m_documentVersionHasBeenSet; }
  template <typename DocumentVersionT = Aws::String>
  void SetDocumentVersion(DocumentVersionT&& value) {
    m_documentVersionHasBeenSet = true;
    m_documentVersion = std::forward<DocumentVersionT>(value);
  }
  template <typename DocumentVersionT = Aws::String>
  UpdateDocumentRequest& WithDocumentVersion(DocumentVersionT&& value) {
    SetDocumentVersion(std::forward<DocumentVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify the document format for the new document version. Systems Manager
   * supports JSON and YAML documents. JSON is the default format.</p>
   */
  inline DocumentFormat GetDocumentFormat() const { return m_documentFormat; }
  inline bool DocumentFormatHasBeenSet() const { return m_documentFormatHasBeenSet; }
  inline void SetDocumentFormat(DocumentFormat value) {
    m_documentFormatHasBeenSet = true;
    m_documentFormat = value;
  }
  inline UpdateDocumentRequest& WithDocumentFormat(DocumentFormat value) {
    SetDocumentFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify a new target type for the document.</p>
   */
  inline const Aws::String& GetTargetType() const { return m_targetType; }
  inline bool TargetTypeHasBeenSet() const { return m_targetTypeHasBeenSet; }
  template <typename TargetTypeT = Aws::String>
  void SetTargetType(TargetTypeT&& value) {
    m_targetTypeHasBeenSet = true;
    m_targetType = std::forward<TargetTypeT>(value);
  }
  template <typename TargetTypeT = Aws::String>
  UpdateDocumentRequest& WithTargetType(TargetTypeT&& value) {
    SetTargetType(std::forward<TargetTypeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_content;

  Aws::Vector<AttachmentsSource> m_attachments;

  Aws::String m_name;

  Aws::String m_displayName;

  Aws::String m_versionName;

  Aws::String m_documentVersion;

  DocumentFormat m_documentFormat{DocumentFormat::NOT_SET};

  Aws::String m_targetType;
  bool m_contentHasBeenSet = false;
  bool m_attachmentsHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_displayNameHasBeenSet = false;
  bool m_versionNameHasBeenSet = false;
  bool m_documentVersionHasBeenSet = false;
  bool m_documentFormatHasBeenSet = false;
  bool m_targetTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
