﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/opensearchserverless/OpenSearchServerlessRequest.h>
#include <aws/opensearchserverless/OpenSearchServerless_EXPORTS.h>
#include <aws/opensearchserverless/model/CollectionType.h>
#include <aws/opensearchserverless/model/EncryptionConfig.h>
#include <aws/opensearchserverless/model/StandbyReplicas.h>
#include <aws/opensearchserverless/model/Tag.h>
#include <aws/opensearchserverless/model/VectorOptions.h>

#include <utility>

namespace Aws {
namespace OpenSearchServerless {
namespace Model {

/**
 */
class CreateCollectionRequest : public OpenSearchServerlessRequest {
 public:
  AWS_OPENSEARCHSERVERLESS_API CreateCollectionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateCollection"; }

  AWS_OPENSEARCHSERVERLESS_API Aws::String SerializePayload() const override;

  AWS_OPENSEARCHSERVERLESS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Name of the collection.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateCollectionRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of collection.</p>
   */
  inline CollectionType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(CollectionType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline CreateCollectionRequest& WithType(CollectionType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Description of the collection.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateCollectionRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An arbitrary set of tags (key–value pairs) to associate with the OpenSearch
   * Serverless collection.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateCollectionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateCollectionRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether standby replicas should be used for a collection.</p>
   */
  inline StandbyReplicas GetStandbyReplicas() const { return m_standbyReplicas; }
  inline bool StandbyReplicasHasBeenSet() const { return m_standbyReplicasHasBeenSet; }
  inline void SetStandbyReplicas(StandbyReplicas value) {
    m_standbyReplicasHasBeenSet = true;
    m_standbyReplicas = value;
  }
  inline CreateCollectionRequest& WithStandbyReplicas(StandbyReplicas value) {
    SetStandbyReplicas(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration options for vector search capabilities in the collection.</p>
   */
  inline const VectorOptions& GetVectorOptions() const { return m_vectorOptions; }
  inline bool VectorOptionsHasBeenSet() const { return m_vectorOptionsHasBeenSet; }
  template <typename VectorOptionsT = VectorOptions>
  void SetVectorOptions(VectorOptionsT&& value) {
    m_vectorOptionsHasBeenSet = true;
    m_vectorOptions = std::forward<VectorOptionsT>(value);
  }
  template <typename VectorOptionsT = VectorOptions>
  CreateCollectionRequest& WithVectorOptions(VectorOptionsT&& value) {
    SetVectorOptions(std::forward<VectorOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the collection group to associate with the collection.</p>
   */
  inline const Aws::String& GetCollectionGroupName() const { return m_collectionGroupName; }
  inline bool CollectionGroupNameHasBeenSet() const { return m_collectionGroupNameHasBeenSet; }
  template <typename CollectionGroupNameT = Aws::String>
  void SetCollectionGroupName(CollectionGroupNameT&& value) {
    m_collectionGroupNameHasBeenSet = true;
    m_collectionGroupName = std::forward<CollectionGroupNameT>(value);
  }
  template <typename CollectionGroupNameT = Aws::String>
  CreateCollectionRequest& WithCollectionGroupName(CollectionGroupNameT&& value) {
    SetCollectionGroupName(std::forward<CollectionGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Encryption settings for the collection.</p>
   */
  inline const EncryptionConfig& GetEncryptionConfig() const { return m_encryptionConfig; }
  inline bool EncryptionConfigHasBeenSet() const { return m_encryptionConfigHasBeenSet; }
  template <typename EncryptionConfigT = EncryptionConfig>
  void SetEncryptionConfig(EncryptionConfigT&& value) {
    m_encryptionConfigHasBeenSet = true;
    m_encryptionConfig = std::forward<EncryptionConfigT>(value);
  }
  template <typename EncryptionConfigT = EncryptionConfig>
  CreateCollectionRequest& WithEncryptionConfig(EncryptionConfigT&& value) {
    SetEncryptionConfig(std::forward<EncryptionConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier to ensure idempotency of the request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateCollectionRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  CollectionType m_type{CollectionType::NOT_SET};

  Aws::String m_description;

  Aws::Vector<Tag> m_tags;

  StandbyReplicas m_standbyReplicas{StandbyReplicas::NOT_SET};

  VectorOptions m_vectorOptions;

  Aws::String m_collectionGroupName;

  EncryptionConfig m_encryptionConfig;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_nameHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_standbyReplicasHasBeenSet = false;
  bool m_vectorOptionsHasBeenSet = false;
  bool m_collectionGroupNameHasBeenSet = false;
  bool m_encryptionConfigHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace OpenSearchServerless
}  // namespace Aws
