﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/medialive/MediaLiveRequest.h>
#include <aws/medialive/MediaLive_EXPORTS.h>
#include <aws/medialive/model/ChannelClass.h>
#include <aws/medialive/model/OutputDestination.h>

#include <utility>

namespace Aws {
namespace MediaLive {
namespace Model {

/**
 * Channel class that the channel should be updated to.<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/UpdateChannelClassRequest">AWS
 * API Reference</a></p>
 */
class UpdateChannelClassRequest : public MediaLiveRequest {
 public:
  AWS_MEDIALIVE_API UpdateChannelClassRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateChannelClass"; }

  AWS_MEDIALIVE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * The channel class that you wish to update this channel to use.
   */
  inline ChannelClass GetChannelClass() const { return m_channelClass; }
  inline bool ChannelClassHasBeenSet() const { return m_channelClassHasBeenSet; }
  inline void SetChannelClass(ChannelClass value) {
    m_channelClassHasBeenSet = true;
    m_channelClass = value;
  }
  inline UpdateChannelClassRequest& WithChannelClass(ChannelClass value) {
    SetChannelClass(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Channel Id of the channel whose class should be updated.
   */
  inline const Aws::String& GetChannelId() const { return m_channelId; }
  inline bool ChannelIdHasBeenSet() const { return m_channelIdHasBeenSet; }
  template <typename ChannelIdT = Aws::String>
  void SetChannelId(ChannelIdT&& value) {
    m_channelIdHasBeenSet = true;
    m_channelId = std::forward<ChannelIdT>(value);
  }
  template <typename ChannelIdT = Aws::String>
  UpdateChannelClassRequest& WithChannelId(ChannelIdT&& value) {
    SetChannelId(std::forward<ChannelIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * A list of output destinations for this channel.
   */
  inline const Aws::Vector<OutputDestination>& GetDestinations() const { return m_destinations; }
  inline bool DestinationsHasBeenSet() const { return m_destinationsHasBeenSet; }
  template <typename DestinationsT = Aws::Vector<OutputDestination>>
  void SetDestinations(DestinationsT&& value) {
    m_destinationsHasBeenSet = true;
    m_destinations = std::forward<DestinationsT>(value);
  }
  template <typename DestinationsT = Aws::Vector<OutputDestination>>
  UpdateChannelClassRequest& WithDestinations(DestinationsT&& value) {
    SetDestinations(std::forward<DestinationsT>(value));
    return *this;
  }
  template <typename DestinationsT = OutputDestination>
  UpdateChannelClassRequest& AddDestinations(DestinationsT&& value) {
    m_destinationsHasBeenSet = true;
    m_destinations.emplace_back(std::forward<DestinationsT>(value));
    return *this;
  }
  ///@}
 private:
  ChannelClass m_channelClass{ChannelClass::NOT_SET};

  Aws::String m_channelId;

  Aws::Vector<OutputDestination> m_destinations;
  bool m_channelClassHasBeenSet = false;
  bool m_channelIdHasBeenSet = false;
  bool m_destinationsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaLive
}  // namespace Aws
