package tls

import (
	"crypto/tls"
)

var (
	// CipherSuites Map of TLS CipherSuites from crypto/tls
	// Available CipherSuites defined at https://pkg.go.dev/crypto/tls/#pkg-constants
	CipherSuites = map[string]uint16{
		`TLS_RSA_WITH_RC4_128_SHA`:                      tls.TLS_RSA_WITH_RC4_128_SHA,
		`TLS_RSA_WITH_3DES_EDE_CBC_SHA`:                 tls.TLS_RSA_WITH_3DES_EDE_CBC_SHA,
		`TLS_RSA_WITH_AES_128_CBC_SHA`:                  tls.TLS_RSA_WITH_AES_128_CBC_SHA,
		`TLS_RSA_WITH_AES_256_CBC_SHA`:                  tls.TLS_RSA_WITH_AES_256_CBC_SHA,
		`TLS_RSA_WITH_AES_128_CBC_SHA256`:               tls.TLS_RSA_WITH_AES_128_CBC_SHA256,
		`TLS_RSA_WITH_AES_128_GCM_SHA256`:               tls.TLS_RSA_WITH_AES_128_GCM_SHA256,
		`TLS_RSA_WITH_AES_256_GCM_SHA384`:               tls.TLS_RSA_WITH_AES_256_GCM_SHA384,
		`TLS_ECDHE_ECDSA_WITH_RC4_128_SHA`:              tls.TLS_ECDHE_ECDSA_WITH_RC4_128_SHA,
		`TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`:          tls.TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA,
		`TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`:          tls.TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA,
		`TLS_ECDHE_RSA_WITH_RC4_128_SHA`:                tls.TLS_ECDHE_RSA_WITH_RC4_128_SHA,
		`TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA`:           tls.TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA,
		`TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`:            tls.TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
		`TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`:            tls.TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
		`TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`:       tls.TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256,
		`TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`:         tls.TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256,
		`TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`:         tls.TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
		`TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`:       tls.TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256,
		`TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`:         tls.TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
		`TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`:       tls.TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384,
		`TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305`:          tls.TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305,
		`TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`:   tls.TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256,
		`TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305`:        tls.TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305,
		`TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`: tls.TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256,
		`TLS_AES_128_GCM_SHA256`:                        tls.TLS_AES_128_GCM_SHA256,
		`TLS_AES_256_GCM_SHA384`:                        tls.TLS_AES_256_GCM_SHA384,
		`TLS_CHACHA20_POLY1305_SHA256`:                  tls.TLS_CHACHA20_POLY1305_SHA256,
		`TLS_FALLBACK_SCSV`:                             tls.TLS_FALLBACK_SCSV,
	}

	// CipherSuitesReversed Map of TLS CipherSuites from crypto/tls
	// Available CipherSuites defined at https://pkg.go.dev/crypto/tls/#pkg-constants
	CipherSuitesReversed = map[uint16]string{
		tls.TLS_RSA_WITH_RC4_128_SHA:                      `TLS_RSA_WITH_RC4_128_SHA`,
		tls.TLS_RSA_WITH_3DES_EDE_CBC_SHA:                 `TLS_RSA_WITH_3DES_EDE_CBC_SHA`,
		tls.TLS_RSA_WITH_AES_128_CBC_SHA:                  `TLS_RSA_WITH_AES_128_CBC_SHA`,
		tls.TLS_RSA_WITH_AES_256_CBC_SHA:                  `TLS_RSA_WITH_AES_256_CBC_SHA`,
		tls.TLS_RSA_WITH_AES_128_CBC_SHA256:               `TLS_RSA_WITH_AES_128_CBC_SHA256`,
		tls.TLS_RSA_WITH_AES_128_GCM_SHA256:               `TLS_RSA_WITH_AES_128_GCM_SHA256`,
		tls.TLS_RSA_WITH_AES_256_GCM_SHA384:               `TLS_RSA_WITH_AES_256_GCM_SHA384`,
		tls.TLS_ECDHE_ECDSA_WITH_RC4_128_SHA:              `TLS_ECDHE_ECDSA_WITH_RC4_128_SHA`,
		tls.TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA:          `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`,
		tls.TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA:          `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`,
		tls.TLS_ECDHE_RSA_WITH_RC4_128_SHA:                `TLS_ECDHE_RSA_WITH_RC4_128_SHA`,
		tls.TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA:           `TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA`,
		tls.TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA:            `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`,
		tls.TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA:            `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`,
		tls.TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256:       `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`,
		tls.TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256:         `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`,
		tls.TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256:         `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`,
		tls.TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256:       `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`,
		tls.TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384:         `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`,
		tls.TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384:       `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`,
		tls.TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256:   `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`,
		tls.TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256: `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`,
		tls.TLS_AES_128_GCM_SHA256:                        `TLS_AES_128_GCM_SHA256`,
		tls.TLS_AES_256_GCM_SHA384:                        `TLS_AES_256_GCM_SHA384`,
		tls.TLS_CHACHA20_POLY1305_SHA256:                  `TLS_CHACHA20_POLY1305_SHA256`,
		tls.TLS_FALLBACK_SCSV:                             `TLS_FALLBACK_SCSV`,
	}
)

// GetCipherName returns the Cipher suite name.
// Available CipherSuites defined at https://pkg.go.dev/crypto/tls/#pkg-constants
func GetCipherName(connState *tls.ConnectionState) string {
	if cipher, ok := CipherSuitesReversed[connState.CipherSuite]; ok {
		return cipher
	}

	return "unknown"
}
