// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A7__F198

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x)xyzw" };
#else
{ R"xyzw(amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_16_0_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1200)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xef52cdb0u, 0xbc672bb6u, 0, 29 }, // ef52cdb0bc672bb6 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1200
    { 0x62230431u, 0xf6a853b0u, 0, 6 }, // 62230431f6a853b0 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0xb279161du, 0x87302897u, 0, 63 }, // b279161d87302897 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0x6fdf4c53u, 0x2630ff47u, 23, 29 }, // 6fdf4c532630ff47 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1200
    { 0xead7a006u, 0xff5eca58u, 23, 6 }, // ead7a006ff5eca58 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0xd9c209ddu, 0x54924ae3u, 23, 46 }, // d9c209dd54924ae3 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1200
    { 0x086e27e3u, 0xdabf41cdu, 23, 63 }, // 086e27e3dabf41cd = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0x5b3d5424u, 0x80ce84deu, 103, 109 }, // 5b3d542480ce84de = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1200
    { 0x3b61f7b8u, 0x3cbc958eu, 103, 126 }, // 3b61f7b83cbc958e = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1200
    { 0xcd9924e8u, 0xf66203cbu, 103, 86 }, // cd9924e8f66203cb = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0xbc7c42c0u, 0xbc93d123u, 103, 143 }, // bc7c42c0bc93d123 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1200
    { 0x5b138befu, 0x8f46831bu, 166, 86 }, // 5b138bef8f46831b = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__64_64__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0x144628dau, 0xba792aa6u, 166, 143 }, // 144628daba792aa6 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_0_T_T_0___gfx120x__P__64_64__CO__wave4_warp4_stg1--Arch_gfx1200
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1200_mod0
{{ 3, 6, 3, 3, 4, 5, 6, 5, 6, 4},
 { 4, 3, 5, 1, 3, 4, 2, 1, 0, 0},
 { 3, 6, 5, 4, 4, 6, 3, 6, 3, 0},
 { 6, 5, 3, 6, 6, 4, 6, 3, 3, 3},
 { 8, 6, 4, 3, 4, 5, 5, 3, 3, 3},
 { 9, 4, 6, 4, 6, 5, 3, 3, 3, 4},
 {10, 5, 6, 6, 6, 3, 3, 5, 3, 3},
 { 7, 5,11, 3, 3, 3, 3, 5, 3, 3},
 {10, 6,11, 5, 6, 3, 3, 6,12, 5},
 {10, 6, 6, 3, 4, 3, 5, 3,12, 3}}
// End of GPU gfx1200_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A7__F198 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 2;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 2;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx1201_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 16 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

