/*
 * Copyright (C) 2024 Linux Studio Plugins Project <https://lsp-plug.in/>
 *           (C) 2024 Vladimir Sadovnikov <sadko4u@gmail.com>
 *
 * This file is part of lsp-plugins-graph-equalizer
 * Created on: 3 авг. 2021 г.
 *
 * lsp-plugins-graph-equalizer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * lsp-plugins-graph-equalizer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with lsp-plugins-graph-equalizer. If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef PRIVATE_PLUGINS_GRAPH_EQUALIZER_H_
#define PRIVATE_PLUGINS_GRAPH_EQUALIZER_H_

#include <lsp-plug.in/plug-fw/plug.h>
#include <lsp-plug.in/plug-fw/core/IDBuffer.h>
#include <lsp-plug.in/dsp-units/ctl/Bypass.h>
#include <lsp-plug.in/dsp-units/filters/Equalizer.h>
#include <lsp-plug.in/dsp-units/util/Analyzer.h>
#include <lsp-plug.in/dsp-units/util/Delay.h>

#include <private/meta/graph_equalizer.h>

namespace lsp
{
    namespace plugins
    {
        /**
         * Graphic Equalizer Plugin Series
         */
        class graph_equalizer: public plug::Module
        {
            public:
                enum eq_mode_t
                {
                    EQ_MONO,
                    EQ_STEREO,
                    EQ_LEFT_RIGHT,
                    EQ_MID_SIDE
                };

            protected:
                enum chart_state_t
                {
                    CS_UPDATE       = 1 << 0,
                    CS_SYNC_AMP     = 1 << 1
                };

                enum fft_position_t
                {
                    FFTP_NONE,
                    FFTP_POST,
                    FFTP_PRE
                };

                typedef struct eq_band_t
                {
                    bool                bSolo;          // Solo
                    uint32_t            nSync;          // Chart state
                    float              *vTrRe;          // Transfer function (real part)
                    float              *vTrIm;          // Transfer function (imaginary part)

                    plug::IPort        *pGain;          // Gain port
                    plug::IPort        *pSolo;          // Solo port
                    plug::IPort        *pMute;          // Mute port
                    plug::IPort        *pEnable;        // Enable port
                    plug::IPort        *pVisibility;    // Filter visibility
                } eq_band_t;

                typedef struct eq_channel_t
                {
                    dspu::Equalizer     sEqualizer;     // Equalizer
                    dspu::Bypass        sBypass;        // Bypass
                    dspu::Delay         sDryDelay;      // Dry delay

                    uint32_t            nSync;          // Chart state
                    float               fInGain;        // Input gain
                    float               fOutGain;       // Output gain
                    eq_band_t          *vBands;         // Bands
                    float              *vIn;            // Input buffer
                    float              *vOut;           // Output buffer
                    float              *vSend;          // Send buffer
                    float              *vReturn;        // Return buffer
                    float              *vInPtr;         // Actual pointer to input data (for eliminatioon of unnecessary memory copies)
                    float              *vExtPtr;        // External (send) data pointer
                    float              *vDryBuf;        // Dry buffer
                    float              *vInBuffer;      // Input buffer (input signal passed to analyzer)
                    float              *vOutBuffer;     // Output buffer
                    float              *vExtBuffer;     // External (send) buffer

                    float              *vTrRe;          // Transfer function (real part)
                    float              *vTrIm;          // Transfer function (imaginary part)

                    plug::IPort        *pIn;            // Input port
                    plug::IPort        *pOut;           // Output port
                    plug::IPort        *pSend;          // Send port
                    plug::IPort        *pReturn;        // Return port
                    plug::IPort        *pInGain;        // Input gain
                    plug::IPort        *pTrAmp;         // Amplitude chart
                    plug::IPort        *pFftInSwitch;   // FFT input switch
                    plug::IPort        *pFftOutSwitch;  // FFT output switch
                    plug::IPort        *pFftExtSwitch;  // FFT external switch
                    plug::IPort        *pFftInMesh;     // FFT input mesh
                    plug::IPort        *pFftOutMesh;    // FFT output mesh
                    plug::IPort        *pFftExtMesh;    // FFT external mesh
                    plug::IPort        *pVisible;       // Visibility flag
                    plug::IPort        *pInMeter;       // Output level meter
                    plug::IPort        *pOutMeter;      // Output level meter
                } eq_channel_t;

            protected:
                inline dspu::equalizer_mode_t   get_eq_mode();
                void                            dump_channel(dspu::IStateDumper *v, const eq_channel_t *c) const;
                static void                     dump_band(dspu::IStateDumper *v, const eq_band_t *b);

            protected:
                dspu::Analyzer      sAnalyzer;      // Analyzer
                eq_channel_t       *vChannels;      // Equalizer channels
                uint32_t            nBands;         // Number of bands
                uint32_t            nMode;          // Equalize mode
                uint32_t            nSlope;         // Slope
                bool                bListen;        // Listen
                bool                bMatched;       // Matched transorm/Bilinear transform flag
                float               fInGain;        // Input gain
                float               fZoom;          // Zoom gain
                float              *vFreqs;         // Frequency list
                uint32_t           *vIndexes;       // FFT indexes
                core::IDBuffer     *pIDisplay;      // Inline display buffer

                plug::IPort        *pEqMode;        // Equalizer mode
                plug::IPort        *pSlope;         // Filter slope
                plug::IPort        *pListen;        // Mid-Side listen
                plug::IPort        *pInGain;        // Input gain
                plug::IPort        *pOutGain;       // Output gain
                plug::IPort        *pBypass;        // Bypass
                plug::IPort        *pReactivity;    // FFT reactivity
                plug::IPort        *pShiftGain;     // Shift gain
                plug::IPort        *pZoom;          // Graph zoom
                plug::IPort        *pBalance;       // Output balance

            protected:
                void                do_destroy();
                void                perform_analysis(size_t samples);

            public:
                explicit graph_equalizer(const meta::plugin_t *metadata, size_t bands, size_t mode);
                virtual ~graph_equalizer() override;

            public:
                virtual void        init(plug::IWrapper *wrapper, plug::IPort **ports) override;
                virtual void        destroy() override;

                virtual void        update_settings() override;
                virtual void        update_sample_rate(long sr) override;
                virtual void        ui_activated() override;

                virtual void        process(size_t samples) override;
                virtual bool        inline_display(plug::ICanvas *cv, size_t width, size_t height) override;

                virtual void        dump(dspu::IStateDumper *v) const override;
        };
    } /* namespace plugins */
} /* namespace lsp */

#endif /* PRIVATE_PLUGINS_GRAPH_EQUALIZER_H_ */
